package be.re.net;

import be.re.io.ReadLineInputStream;
import be.re.util.Array;
import be.re.util.UncheckedCapsule;
import java.io.File;
import java.io.IOException;
import java.io.UnsupportedEncodingException;
import java.net.MalformedURLException;
import java.net.URL;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.Iterator;
import java.util.List;
import java.util.ResourceBundle;
import java.util.StringTokenizer;



public class Util

{

  private final static String[]	COMPOSED_SCHEMES =
    {"jar", "jms", "jndi"};
  private final static char[]	MARK =
    {'-', '_', '.', '!', '~', '*', '\'', '(', ')'};
  private final static char[]	PCHAR_SPECIALS =
    {':', '@', '&', '=', '+', '$', ','};
  private final static char[]	RESERVED =
    {';', '/', '?', ':', '@', '&', '=', '+'};
  private static ResourceBundle	bundle = null;



  public static void
  addUrlHandlers()
  {
    String	value = System.getProperty("java.protocol.handler.pkgs");

    if (value == null || value.indexOf("be.re.net") == -1)
    {
      System.setProperty
      (
        "java.protocol.handler.pkgs",
        "be.re.net" + (value == null || value.equals("") ? "" : ("|" + value))
      );
    }
  }



  private static boolean
  compareBytes(byte[] b1, byte[] b2, int length)
  {
    for (int i = 0; i < length; ++i)
    {
      if (b1[i] != b2[i])
      {
        return false;
      }
    }

    return true;
  }



  public static String
  displayUrl(URL url)
  {
    return unescapeUriSpecials(url.toString());
  }



  public static String
  escapeUriEscapes(String segment)
  {
    return
      escapeUriPart
      (
        segment,
        new TestChar()
        {
          public boolean
          test(char c)
          {
            return c != '%' && (isUriChar(c) || isPChar(c));
          }
        }
      );
  }



  private static String
  escapeUriPart(String s, TestChar t)
  {
    byte[]	bytes = null;
    String	result = "";

    try
    {
      bytes = s.getBytes("UTF-8");
    }

    catch (UnsupportedEncodingException e)
    {
      throw new UncheckedCapsule(e); // Would be a bug.
    }

    for (int i = 0; i < bytes.length; ++i)
    {
      result +=
        !t.test((char) (0xff & bytes[i])) ?
          ("%" + Integer.toHexString((int) 0xff & bytes[i])) :
          new String(bytes, i, 1);
    }

    return result;
  }



  public static String
  escapeUriPathSegment(String segment)
  {
    return
      escapeUriPart
      (
        segment,
        new TestChar()
        {
          public boolean
          test(char c)
          {
            return isPChar(c) || c == ';' /* For parameters. */;
          }
        }
      );
  }



  public static String
  escapeUriPathSegments(String path)
  {
    String		result = "";
    StringTokenizer	tokenizer = new StringTokenizer(path, "/");

    while (tokenizer.hasMoreTokens())
    {
      result +=
        (result.equals("") ? "" : "/") +
          escapeUriPathSegment(tokenizer.nextToken());
    }

    return
      (path.length() > 0 && path.charAt(0) == '/' ? "/" : "") + result +
        (path.length() > 1 && path.charAt(path.length() - 1) == '/' ? "/" : "");
  }



  public static String
  escapeUriQueryString(String queryString)
  {
    return
      escapeUriPart
      (
        queryString,
        new TestChar()
        {
          public boolean
          test(char c)
          {
            return isUriChar(c);
          }
        }
      );
  }



  public static String
  escapeUriReference(String reference)
  {
    return
      escapeUriPart
      (
        reference,
        new TestChar()
        {
          public boolean
          test(char c)
          {
            return isUriChar(c);
          }
        }
      );
  }



  private static URL
  escapedComposedUrl(String url) throws MalformedURLException
  {
    return
      new URL
      (
        url.substring(0, url.indexOf(':')) + ":" +
          escapedUrl(extractSubUrl(url).toString()).toString() + "!/" +
          escapeUriPathSegments(extractComposedUrlEntry(url)).toString()
      );
  }



  private static File
  escapedFile(File file)
  {
    return
      new File
      (
        escapeUriPathSegments(file.getAbsolutePath().replace('\\', '/'))
      );
  }



  public static URL
  escapedUrl(String url) throws MalformedURLException
  {
    int	colon = url.indexOf(':');

    if (colon != -1 && isComposedUrl(url))
    {
      return escapedComposedUrl(url);
    }

    int	pathStart = 0;

    if (colon != -1)
    {
      pathStart =
        url.indexOf
        (
          '/',
          colon +
            (
              url.length() > colon + 2 &&
                url.substring(colon + 1, colon + 3).equals("//") ?
                3 : 1
            )
        );
    }

    if (pathStart == -1)
    {
      return new URL(url);
    }

    int	queryStart = -1;
    int	referenceStart = -1;

    if ("http".equals(colon != -1 ? url.substring(0, colon) : null))
    {
      queryStart = url.indexOf('?', pathStart);
      referenceStart = url.indexOf('#', queryStart);
    }

    return
      new URL
      (
        url.substring(0, pathStart) +
        escapeUriPathSegments
        (
          url.substring(pathStart, queryStart != -1 ? queryStart : url.length())
        ) +
        (
          queryStart != -1 ?
            (
              "?" +
                escapeUriQueryString
                (
                  url.substring
                  (
                    queryStart + 1,
                    referenceStart != -1 ? referenceStart : url.length()
                  )
                )
            ) : ""
        ) +
        (
          referenceStart != -1 ?
            ("#" + escapeUriReference(url.substring(referenceStart + 1))) : ""
        )
      );
  }



  public static String
  extractComposedUrlEntry(URL url)
  {
    int	index = url.getFile().indexOf("!/");

    return
      index == -1 || index + 2 == url.getFile().length() ?
        "" : url.getFile().substring(index + 2);
  }



  public static String
  extractComposedUrlEntry(String url)
  {
    int	index = url.indexOf("!/");

    return
      index == -1 || index + 2 == url.length() ? "" : url.substring(index + 2);
  }



  public static URL
  extractSubUrl(URL url) throws MalformedURLException
  {
    if (!isComposedUrl(url))
    {
      throw new MalformedURLException("No sub-URL");
    }

    return
      new URL
      (
        url.toString().substring
        (
          url.getProtocol().length() + 1, url.toString().indexOf("!/")
        )
      );
  }



  public static URL
  extractSubUrl(String url) throws MalformedURLException
  {
    return new URL(url.substring(url.indexOf(':') + 1, url.indexOf("!/")));
  }



  public static URL
  fileToUrl(File file)
  {
    try
    {
      return
        escapedFile
        (
          new File(escapeUriEscapes(file.getAbsolutePath().replace('\\', '/')))
        ).toURL();
    }

    catch (MalformedURLException e)
    {
      throw new UncheckedCapsule(e);
        // We made sure the path can be parsed as an URL by escaping it.
    }
  }



  public static String
  getFTPFilename(URL url)
  {
    // According to RFC 959 the file system conventions of the involved
    // server must be followed. The applicable information is, however, not
    // available anymore in the URL, because there the URI rules are followed.
    // Therefore, we assure to have forward slashes, which are widely accepted.

    return urlToFile(url).getPath().replace('\\', '/');
  }



  public static String
  getLastPathSegment(URL url)
  {
    return
      getLastPathSegment
      (
        isComposedUrl(url) ?
          url.getFile().substring(url.getFile().indexOf("!/") + 1) :
          url.getFile()
      );
  }



  /**
   * A trailing slash is ignored.
   */

  public static String
  getLastPathSegment(String path)
  {
    // We're not interested in the last character if it is a slash.

    path = path.replace('\\', '/');

    return path.substring(path.lastIndexOf('/', path.length() - 2) + 1);
  }



  static String
  getResource(String key)
  {
    if (bundle == null)
    {
      bundle = ResourceBundle.getBundle("be.re.net.Res");
    }

    return bundle.getString(key);
  }



  private static String
  getSegments(String path) throws MalformedURLException
  {
    List		segments = new ArrayList();
    StringTokenizer	tokenizer = new StringTokenizer(path, "/");

    while (tokenizer.hasMoreTokens())
    {
      String	token = tokenizer.nextToken();

      if (!token.equals("."))
      {
        if (token.equals(".."))
        {
          if (segments.size() == 0)
          {
            throw
              new MalformedURLException
              (
                "URL specification goes outside of its root"
              );
          }

          segments.remove(segments.size() - 1);
        }
        else
        {
          segments.add(token);
        }
      }
    }

    String	result = "";

    for
    (
      Iterator i = segments.iterator();
      i.hasNext();
      result += (result.equals("") ? "" : "/") + (String) i.next()
    );

    return result;
  }



  public static long
  httpDate(String s)
  {
    String[]	patterns =
      new String[]
      {
        "EEE, dd MMM yyyy HH:mm:ss 'GMT'", // RFC1123-date
        "EEEE, dd-MMM-yy HH:mm:ss 'GMT'", // RFC850-date
        "EEE MMM dd HH:mm:ss yyyy", // ASCTIME-date
        "EEE MMM  d HH:mm:ss yyyy" // ASCTIME-date
      };

    for (int i = 0; i < patterns.length; ++i)
    {
      try
      {
        Date	date = new SimpleDateFormat(patterns[i]).parse(s);

        if (date != null)
        {
          return date.getTime();
        }
      }

      catch (ParseException e)
      {
      }
    }

    return 0;
  }



  private static boolean
  inArray(char[] array, char c)
  {
    for (int i = 0; i < array.length; ++i)
    {
      if (c == array[i])
      {
        return true;
      }
    }

    return false;
  }



  public static boolean
  isComposedUrl(URL url)
  {
    return Array.inArray(COMPOSED_SCHEMES, url.getProtocol());
  }



  public static boolean
  isComposedUrl(String url) throws MalformedURLException
  {
    return isComposedUrl(new URL(url));
  }



  private static boolean
  isPChar(char c)
  {
    return c == '%' || isUnreserved(c) || inArray(PCHAR_SPECIALS, c);
  }



  private static boolean
  isUnreserved(char c)
  {
    return
      (c >= 'a' && c <= 'z') || (c >= 'A' && c <= 'Z') ||
        (c >= '0' && c <= '9') || inArray(MARK, c);
  }



  private static boolean
  isUriChar(char c)
  {
    return c == '%' || isUnreserved(c) || inArray(RESERVED, c);
  }



  public static boolean
  isUrl(String s)
  {
    try
    {
      return s != null && new URL(s) != null;
    }

    catch (MalformedURLException e)
    {
      return false;
    }
  }



  public static Headers
  readHeaders(ReadLineInputStream in) throws IOException
  {
    Headers	headers = new Headers();
    byte[]	line;

    while ((line = in. readLine()) != null && line.length > 0)
    {
      String	s = new String(line);

      if (s.indexOf(':') != -1)
      {
        headers.add
        (
          s.substring(0, s.indexOf(':')).trim(),
          s.substring(s.indexOf(':') + 1).trim()
        );
      }
    }

    return headers;
  }



  public static URL
  resolvePath(URL url) throws MalformedURLException
  {
    if (isComposedUrl(url))
    {
      try
      {
        String	segments = getSegments(extractComposedUrlEntry(url));

        segments =
          (segments.length() == 0 || segments.charAt(0) != '/' ? "/" : "") +
            segments;

        return
          new URL
          (
            url.getProtocol() + ":" +
              resolvePath(extractSubUrl(url)).toString() + "!" +
              segments +
              (
                segments.charAt(segments.length() - 1) != '/' &&
                  url.toString().charAt(url.toString().length() - 1) == '/' ?
                  "/" : ""
              )
          );
      }

      catch (MalformedURLException e)
      {
        // This considers a jar URL to be an ordinary directory.

        return
          resolvePath
          (
            new URL
            (
              extractSubUrl(url).toString() + "/" + extractComposedUrlEntry(url)
            )
          );
      }
    }

    String	segments = getSegments(url.getFile());

    segments =
      (segments.length() == 0 || segments.charAt(0) != '/' ? "/" : "") +
        segments;

    return
      new URL
      (
        new URL
        (
          url.getProtocol() + ":" +
            (url.getAuthority() != null ? "//" + url.getAuthority() : "")
        ),
        segments +
          (
            segments.charAt(segments.length() - 1) != '/' &&
              url.toString().charAt(url.toString().length() - 1) == '/' ?
              "/" : ""
          ) + (url.getRef() != null ? ("#" + url.getRef()) : "")
      );
  }



  public static URL
  setUserInfo(URL url, String userInfo) throws MalformedURLException
  {
    url = stripUserInfo(url);

    return
      new URL
      (
        url.getProtocol() + ":" +
          (url.getHost() != null ? "//" + userInfo + "@" + url.getHost() : "") +
          (url.getPort() != -1 ?  (":" + String.valueOf(url.getPort())) : "") +
          url.getFile() + (url.getRef() != null ? ("#" + url.getRef()) : "")
      );
  }



  public static URL
  stripUserInfo(URL url)
  {
    try
    {
      return
        new URL
        (
          url.getProtocol() + ":" +
            (url.getHost() != null ? "//" + url.getHost() : "") +
            (url.getPort() != -1 ?  (":" + String.valueOf(url.getPort())) : "")
            + url.getFile() + (url.getRef() != null ? ("#" + url.getRef()) : "")
        );
    }

    catch (MalformedURLException e)
    {
      throw new UncheckedCapsule(e); // Would be a bug.
    }
  }



  public static String
  unescapeUriSpecials(String s)
  {
    int		count = 0;
    byte[]	result = new byte[s.length()];

    for (int i = 0; i < s.length(); ++i)
    {
      if (s.charAt(i) == '%' && s.length() - i > 2)
      {
        result[count++] =
          (byte) Integer.parseInt(s.substring(i + 1, i + 3), 16);
        i += 2;
      }
      else
      {
        result[count++] = (byte) s.charAt(i);
      }
    }

    try
    {
      return new String(result, 0, count, "UTF-8");
    }

    catch (UnsupportedEncodingException e)
    {
      throw new UncheckedCapsule(e); // Would be a bug.
    }
  }



  public static File
  urlToFile(URL url)
  {
    return new File(unescapeUriSpecials(url.getFile()));
  }



  public static User
  userFromUrl(URL url)
  {
    User	user = new BasicUser();
    String	userInfo = url.getUserInfo();

    if (userInfo == null)
    {
      return user;
    }

    StringTokenizer	tokenizer = new StringTokenizer(userInfo, ":");

    if (tokenizer.hasMoreTokens())
    {
      user.setUsername(tokenizer.nextToken());
    }

    if (tokenizer.hasMoreTokens())
    {
      user.setPassword(tokenizer.nextToken());
    }

    return user;
  }



  private interface TestChar
  {
    public boolean	test	(char c);
  }

} // Util
