package be.re.xml.sax;

import be.re.io.StreamConnector;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.HashMap;
import java.util.Map;
import java.util.StringTokenizer;
import org.xml.sax.EntityResolver;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;



public class CatalogResolver implements EntityResolver

{

  // Alphabet.

  final static int		SINGLE_QUOTE = 0;
  final static int		DOUBLE_QUOTE = 1;
  final static int		OTHER = 2;
  final static int		SPACE = 3;
  final static int		WHITE = 4;
  final static int		EOF = 5;

  // States.

  final static int		TYP = 0;
  final static int		SQ1 = 1;
  final static int		DQ1 = 2;
  final static int		ID1 = 3;
  final static int		SQ2 = 4;
  final static int		DQ2 = 5;
  final static int		ERR = 6;

  final static int[][][]	FSM =
    {
      {{SQ1, 1}, {DQ1, 1}, {TYP, 0}, {TYP, 0}, {TYP, 0}, {TYP, 0}}, // TYP
      {{ID1, 1}, {SQ1, 0}, {SQ1, 0}, {SQ1, 0}, {ERR, 0}, {ERR, 0}}, // SQ1
      {{DQ1, 0}, {ID1, 1}, {DQ1, 0}, {DQ1, 0}, {ERR, 0}, {ERR, 0}}, // DQ1
      {{SQ2, 1}, {DQ2, 1}, {ERR, 0}, {ID1, 0}, {ID1, 0}, {ERR, 0}}, // ID1
      {{TYP, 1}, {SQ2, 0}, {SQ2, 0}, {SQ2, 0}, {ERR, 0}, {ERR, 0}}, // SQ2
      {{DQ2, 0}, {TYP, 1}, {DQ2, 0}, {DQ2, 0}, {ERR, 0}, {ERR, 0}}  // DQ2
    };

  private URL	catalogUrl;
  private Map	publicIdentifiers = new HashMap();
  private Map	systemIdentifiers = new HashMap();



  public
  CatalogResolver(URL catalogUrl) throws IOException
  {
    this.catalogUrl = catalogUrl;
    load();
  }



  private static int
  category(int c)
  {
    return
      c == '\'' ?
        SINGLE_QUOTE :
        (
          c == '\"' ?
            DOUBLE_QUOTE :
            (
              c == ' ' ?
                SPACE :
                (
                  c == '\t' || c == '\n' || c == '\r' ?
                    WHITE : OTHER
                )
            )
        );
  }



  private static void
  error(int in, int line) throws IOException
  {
    if (in == EOF)
    {
      throw
        new IOException
        (
          "Line " + String.valueOf(line) + ": premature end of file"
        );
    }

    if (in == WHITE)
    {
      throw
        new IOException
        (
          "Line " + String.valueOf(line) +
            ": \\t, \\n and \\r are not allowed in an identifier"
        );
    }

    if (in == OTHER)
    {
      throw
        new IOException
        (
          "Line " + String.valueOf(line) + ": white space expected"
        );
    }
  }



  private static String
  getTypeToken(char[] c, int off, int len, int line) throws IOException
  {
    StringTokenizer	tokenizer =
      new StringTokenizer(new String(c, off, len), " \t\n\r");

    if (!tokenizer.hasMoreTokens())
    {
      throw
        new IOException
        (
          "Line " + String.valueOf(line) + ": PUBLIC or SYSTEM expected"
        );
    }

    String	token = tokenizer.nextToken();

    if (!token.equals("PUBLIC") && !token.equals("SYSTEM"))
    {
      throw
        new IOException
        (
          "Line " + String.valueOf(line) + ": PUBLIC or SYSTEM expected"
        );
    }

    return token;
  }



  private void
  load() throws IOException
  {
    ByteArrayOutputStream	out = new ByteArrayOutputStream();

    StreamConnector.copy(catalogUrl.openStream(), out);

    char[]	c = new String(out.toByteArray(), "ASCII").toCharArray();
    String	from = null;
    int		line = 1;
    int		position = 0;
    int		state = TYP;
    String	type = null;

    for (int i = 0; i < c.length; ++i)
    {
      int[]	next = FSM[state][category(c[i])];

      if (next[0] == ERR)
      {
        error(category(c[i]), line);
      }

      if (next[1] == 1)
      {
        Map	map;

        switch (state)
        {
          case TYP:
            type = getTypeToken(c, position, i - position, line);
            break;

          case SQ1: case DQ1:
            from = new String(c, position, i - position);
            break;

          case SQ2: case DQ2:
            map = type.equals("PUBLIC") ? publicIdentifiers : systemIdentifiers;
            map.put(from, new String(c, position, i - position));
            break;
        }

        position = i + 1;
      }

      state = next[0];

      if (c[i] == '\n')
      {
        ++line;
      }
    }

    if (FSM[state][EOF][0] == ERR)
    {
      error(EOF, line);
    }
  }



  public InputSource
  resolveEntity(String publicId, String systemId)
    throws IOException, SAXException
  {
    return
      publicId != null && publicIdentifiers.get(publicId) != null ?
        resolveUrl((String) publicIdentifiers.get(publicId)) :
        (
          systemId != null && systemIdentifiers.get(systemId) != null ?
            resolveUrl((String) systemIdentifiers.get(systemId)) : null
        );
  }



  private InputSource
  resolveUrl(String url) throws MalformedURLException
  {
    return new InputSource(new URL(catalogUrl, url).toString());
  }

} // CatalogResolver
